# Copyright (c) 2012 Barbara Schmid.
# All rights reserved. This program and the accompanying materials
# are made available under the terms of the GNU Public License v3.0
# which accompanies this distribution, and is available at
# http://www.gnu.org/licenses/gpl.html
# 
# Contributors:
#     Barbara Schmid - initial API and implementation

from pyramid.response import Response
from pyramid.view import view_config
from pyramid.httpexceptions import (
    HTTPOk,
    )

from sqlalchemy.exc import DBAPIError

from .models import (
    DBSession,
    MyModel,
    AnsweredQuestionnaire,
    Log,
    )

from json import dumps, loads
from time import time, localtime, strftime, mktime, strptime
from codecs import open
from hashlib import md5
import re
from cStringIO import StringIO
from zipfile import ZipFile
import os

@view_config(route_name='home', renderer='templates/mytemplate.pt')
def my_view(request):
    try:
        one = DBSession.query(MyModel).filter(MyModel.name=='one').first()
    except DBAPIError:
        return Response(conn_err_msg, content_type='text/plain', status_int=500)
    return {'one':one, 'project':'qserve'}

conn_err_msg = """\
Pyramid is having a problem using your SQL database.  The problem
might be caused by one of the following things:

1.  You may need to run the "initialize_qserve_db" script
    to initialize your database tables.  Check your virtual 
    environment's "bin" directory for this script and try to run it.

2.  Your database server may not be running.  Check that the
    database server referred to by the "sqlalchemy.url" setting in
    your "development.ini" file is running.

After you fix the problem, please restart the Pyramid application to
try it again.
"""

@view_config(route_name='questionnaires', renderer='json')
def questionnaire_view(request):
    timestamp=int(request.matchdict["timestamp"])
    if request.method == "GET":
        xmls=[]
        for fname in os.listdir("."):
            if not fname.endswith(".xml"):
                continue
            filetime=os.path.getmtime(fname)
            #print filetime, timestamp
            if filetime<timestamp:
                continue
            #print(int(filetime))
            f=open(fname, 'r', 'utf8')
            xml=f.read()
            xmls.append(xml)
        return {'questionnaires':xmls, "timestamp":int(time())}
    else:
        xmls = request.json_body
        valid_xmls = []
        for i in xmls.keys():
	    fname = "questionnaire_" + md5(i.encode("utf8")).hexdigest() + ".xml"
            valid_xmls.append(fname)
            try:    oldcontent = open(fname, 'r', 'utf8').read()
            except: oldcontent = ""
            if xmls[i] != oldcontent:
                #print "content changed"
                open(fname, "w", 'utf8').write(xmls[i])
        have_xmls = []
        for fname in os.listdir("."):
            if fname.endswith(".xml"):
                have_xmls.append(fname)
        
        delete_xmls = set(have_xmls) - set(valid_xmls)
        for fname in delete_xmls:
            os.remove(fname)




@view_config(route_name='answers', renderer='json')
def answers_view(request):
    session=DBSession()
    for answer in request.json_body:
        aq=AnsweredQuestionnaire(answer["userId"], answer["timestamp"], answer["questionnaire"], dumps(answer["answers"], ensure_ascii=False))
        session.add(aq)
    session.flush()
    return {}

@view_config(route_name='logs', renderer='json')
def logs_view(request):
    session=DBSession()
    for log in request.json_body:
        lg=Log(log["userId"], log["timestamp"], log["duration"], log["activity"], log["packagename"], log["topackagename"], log["location"])
        session.add(lg)
    session.flush()
    return {}

@view_config(route_name='database', renderer='json')
def database_view(request):
    result={"answers":[], "logs":[]}
    answers=DBSession.query(AnsweredQuestionnaire).all()
    for a in answers:
        result["answers"].append({
            "userId":a.userId,
            "timestamp":strftime("%Y-%m-%d %H:%M:%S", localtime(a.timestamp/1000.)),
            "questionnaire":a.questionnaire,
            "answers":a.answers,
            })    
    logs=DBSession.query(Log).all()
    for a in logs:
        result["logs"].append({
            "userId":a.userId,
            "timestamp":strftime("%Y-%m-%d %H:%M:%S", localtime(a.timestamp/1000.)),
            "duration":a.duration,
            "activity":a.activity,
            "packagename":a.packagename,
            "topackagename":a.topackagename,
            "location":a.location,
            })
    return result

#
# EVALUATION
#

@view_config(route_name='users', renderer='json')
def users_view(request):
    result={"users":[]}
    answers=DBSession.query(AnsweredQuestionnaire).all()
    userIds = set([ a.userId for a in answers ])
    for u in userIds:
    	result["users"].append({'userId':u})
    return result

def createTimeline(userId):
    answers=DBSession.query(AnsweredQuestionnaire).filter(AnsweredQuestionnaire.userId==userId).all()
    logs=DBSession.query(Log).filter(Log.userId==userId).all()

    tsa = set([ t.timestamp for t in answers ])
    tsl = set([ t.timestamp for t in logs ])
    ts = sorted(tsa | tsl)

    timeline = []
    for k in ts:
	for a in [ t for t in logs if t.timestamp == k ]:
		timeline.append({
		    "timestamp":strftime("%Y-%m-%d %H:%M:%S", localtime(a.timestamp/1000.)),
		    "questionnaire":'',
		    "answers":'',
		    "duration":a.duration,
		    "activity":a.activity,
		    "packagename":a.packagename,
		    "topackagename":a.topackagename,
		    "location":a.location,
		})
	for a in [ t for t in answers if t.timestamp == k ]:
		timeline.append({
		    "timestamp":strftime("%Y-%m-%d %H:%M:%S", localtime(a.timestamp/1000.)),
		    "questionnaire":a.questionnaire,
		    "answers":a.answers,
		    "duration":'',
		    "activity":'',
		    "packagename":'',
		    "topackagename":'',
		    "location":'',
		})
    
    if len(timeline) != len(answers)+len(logs):
    	raise Error

    return timeline


def compressTimeline(timeline):
    ct = []

    # Merge multiple activity changes
    previous = timeline[0]
    for i in range(1, len(timeline)):
    	if previous['packagename'] == previous['topackagename'] and previous['packagename'] == timeline[i]['packagename']:
		previous['duration'] += timeline[i]['duration']
		previous['activity'] += ', ' + timeline[i]['activity']
		previous['topackagename'] = timeline[i]['topackagename']
	else:
		ct.append(previous)
		previous = timeline[i]
    ct.append(previous)

    timeline = ct
    ct = []
    # Merge close points in time
    previous = timeline[0]
    nonmailapps = [ "", "com.questionnaire", "com.facebook.katana" ]
    for i in range(1, len(timeline)):
    	if previous['duration'] and ((previous['packagename'] not in nonmailapps and timeline[i]['packagename'] not in nonmailapps) or (previous['packagename']==timeline[i]['packagename'])) and mktime(strptime(timeline[i]['timestamp'], '%Y-%m-%d %H:%M:%S'))-(mktime(strptime(previous['timestamp'], '%Y-%m-%d %H:%M:%S'))+previous['duration']/1000.)<=60:
		previous['duration'] = int(mktime(strptime(timeline[i]['timestamp'], '%Y-%m-%d %H:%M:%S')) + round(timeline[i]['duration']/1000.) - mktime(strptime(previous['timestamp'], '%Y-%m-%d %H:%M:%S')))*1000
		#print previous['duration']
		#print mktime(strptime(timeline[i]['timestamp'], '%Y-%m-%d %H:%M:%S')), timeline[i]['duration']/1000., mktime(strptime(previous['timestamp'], '%Y-%m-%d %H:%M:%S'))
		previous['activity'] += ', (merge) ' + timeline[i]['activity']
		previous['topackagename'] += ', (merge) ' + timeline[i]['topackagename']
	else:
		ct.append(previous)
		previous = timeline[i]
    ct.append(previous)

    return ct


@view_config(route_name='timeline', renderer='json')
def timeline_view(request):
    userId=request.matchdict["userId"]
   
    #result={"timeline": createTimeline(userId)}
    result={"timeline": compressTimeline(createTimeline(userId))}
    return result



@view_config(route_name='evaluation', renderer='json')
def evaluation_view(request):
    userId=request.matchdict["userId"]
    result={"evaluation":{}}
    
    answers=DBSession.query(AnsweredQuestionnaire).all()
    userIds = set([ a.userId for a in answers ])

    # Create a mapping between userid and mode
    useridMap = {'EVENTTRIGGERED':[], 'INTERVALTRIGGERED':[], 'VOLUNTARY':[]}

    # SystemInfo
    sysinfo = []
    sysinfo_data = [ a for a in answers if a.questionnaire == "SystemInfo" ]
    ts = sorted([ a.timestamp for a in sysinfo_data ])
    for k in ts:
    	for a in [ t for t in sysinfo_data if t.timestamp == k ]:
	    ans = loads(a.answers)
	    # THIS FIXES BROKEN ENTRY, REMOvE
	    if not ans.has_key('mode'): continue #ans['mode']='VOLUNTARY'
	    useridMap[ans['mode']].append(a.userId)
	    sysinfo.append({
			"userId":a.userId,
			"timestamp":strftime("%Y-%m-%d %H:%M:%S", localtime(a.timestamp/1000.)),
			"questionnaire":a.questionnaire,
			"answers":a.answers,
			})
    result['evaluation']['systeminfo'] = sysinfo
    

    # Anfangs-Fragebogen
    aq = {}
    aq_data = [ a for a in answers if a.questionnaire == "Anfangs - Fragebogen" ]
    for k in useridMap.keys():
    	aq[k] = []
	for u in useridMap[k]:
    		for a in [ t for t in aq_data if t.userId == u ]:
			aq[k].append({
			"userId":a.userId,
			"timestamp":strftime("%Y-%m-%d %H:%M:%S", localtime(a.timestamp/1000.)),
			"questionnaire":a.questionnaire,
			"answers":a.answers,
			})
    result['evaluation']['aq'] = aq
    
    
    # End-Fragebogen
    eq = {}
    eq_data = [ a for a in answers if a.questionnaire == "End - Fragebogen" ]
    for k in useridMap.keys():
    	eq[k] = []
	for u in useridMap[k]:
    		for a in [ t for t in eq_data if t.userId == u ]:
			eq[k].append({
			"userId":a.userId,
			"timestamp":strftime("%Y-%m-%d %H:%M:%S", localtime(a.timestamp/1000.)),
			"questionnaire":a.questionnaire,
			"answers":a.answers,
			})
    print eq
    result['evaluation']['eq'] = eq
    return result



@view_config(route_name='createapp', renderer='json')
def createapp_view(request):
    try:
        modvars = dict(request.params)
    except:
        modvars = {}
    modvars['private String serverIP'] = '"' + request.application_url + '"'
    print modvars

    sourceIO = StringIO()
    sourceIO.write(file('Questionnaire.zip', 'rb').read())
    source = ZipFile(sourceIO, 'a')
    code = source.read('Questionnaire/src/com/questionnaire/QuestionnaireService.template')
    for key in modvars.keys():
        code = re.sub(r'' + key + ' =.*', key + ' = ' + modvars[key] + ';', code, 0, re.MULTILINE)
	#m = re.findall(key + '.*', code)
	#print m
    source.writestr('Questionnaire/src/com/questionnaire/QuestionnaireService.java', code)

    for fname in os.listdir("."):
        if not fname.endswith(".xml"): continue
        f=open(fname, 'r', 'utf8')
        xml=f.read()
        source.writestr('Questionnaire/assets/questionnaires/' + fname, xml)

    source.close()
    sourceIO.reset()

    response = Response(body=sourceIO.read(), content_type='application/octet-stream')
    response.content_disposition = 'attachment; filename="Questionnaire.zip"'
    return response


