/*******************************************************************************
 * Copyright (c) 2012 Barbara Schmid.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Barbara Schmid - initial API and implementation
 ******************************************************************************/
package com.questionnaire;

import java.io.StringReader;
import java.util.ArrayList;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

import android.app.Activity;
import android.app.AlertDialog;
import android.app.Dialog;
import android.content.ComponentName;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.ServiceConnection;
import android.graphics.Color;
import android.os.Bundle;
import android.os.IBinder;
import android.os.Message;
import android.os.Messenger;
import android.os.RemoteException;
import android.text.Editable;
import android.text.InputFilter;
import android.text.InputType;
import android.text.Spanned;
import android.text.TextWatcher;
import android.util.Log;
import android.view.Gravity;
import android.view.KeyEvent;
import android.view.View;
import android.view.inputmethod.EditorInfo;
import android.view.inputmethod.InputMethodManager;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.Button;
import android.widget.CheckBox;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.LinearLayout.LayoutParams;
import android.widget.RadioButton;
import android.widget.RadioGroup;
import android.widget.RelativeLayout;
import android.widget.ScrollView;
import android.widget.SeekBar;
import android.widget.SeekBar.OnSeekBarChangeListener;
import android.widget.Spinner;
import android.widget.TextView;
import android.widget.ToggleButton;

// Displays chosen questionnaire and sends answers to QuestionnaireService 
public class QuestionnaireAsker extends Activity {
	private class State {
		// The current answer
		public String currentAnswer = "";

		// The list of all answers
		public JSONArray answers = new JSONArray();

		// Count of checked checkBoxes
		public int checkBoxCount;

		// The checkBox click handlers require access to all available checkBoxes 
		public CheckBox[] checkBoxes;

		// Information about the current questionnaire.
		public String questionnaireName = "";
		public Node node = null;		
	}

	private State state;

	// Dialog window for pushed back button
	private static final int DIALOG_BACK_ALERT_ID = 0;

	// Enables to send messages to QuestionnaireService as soon as questionnaire is answered
	private Messenger messenger = null;	
	private ServiceConnection conn = new ServiceConnection() {
		public void onServiceConnected(ComponentName className, IBinder binder) {
			messenger = new Messenger(binder);
		}
		public void onServiceDisconnected(ComponentName className) {
			messenger = null;
		}
	};

	@Override
	public void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		Log.d("Activity", "onCreate");

		// Retrieves the current state and answers after screen orientation change
		State lastState = (State) getLastNonConfigurationInstance();
		if(lastState != null) {
			state = lastState;
		} else {
			Intent myIntent = getIntent();
			String xml = myIntent.getExtras().getString("xml");

			state = new State();

			state.answers = new JSONArray();
			loadQuestionnaire(xml);
		}
		// Show current question
		displayCurrentQuestion();
	}
	
	// Returns the current state
	@Override
	public Object onRetainNonConfigurationInstance() {
		return state;
	}

	@Override
	public void onNewIntent(Intent intent) {
		Log.d("Activity", "onNewIntent");
		String xml = intent.getExtras().getString("xml");

		state.answers = new JSONArray();
		loadQuestionnaire(xml);	
		displayCurrentQuestion();
	}

	@Override
	protected void onResume() {
		super.onResume();
		Log.d("Activity", "onResume");

		Intent intent = new Intent(getApplicationContext(), QuestionnaireService.class);
		bindService(intent, conn, 0);
	}

	@Override
	protected void onPause() {
		unbindService(conn);
		Log.d("Activity", "onPause");
		super.onPause();
	}

	@Override
	public void onStop() {
		Log.d("Activity", "onStop");
		super.onStop();
	}

	@Override
	public void onDestroy() {
		Log.d("Activity", "onDestroy");
		super.onDestroy();
	}

	// Shows current question or thank you screen
	public void displayCurrentQuestion() {
		LinearLayout outerLayout = new LinearLayout(this);
		outerLayout.setOrientation(LinearLayout.VERTICAL);
		outerLayout.setLayoutParams(new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT));

		// Final screen and upload to service
		if (state.node == null) {
			TextView question = new TextView(this);
			question.setText(getResources().getString(R.string.quest_thankYou));
			question.setTextSize(35);
			question.setTextColor(Color.WHITE);
			question.setGravity(Gravity.CENTER_HORIZONTAL);
			outerLayout.addView(question);

			LinearLayout backLayout= new LinearLayout(this);
			backLayout.setOrientation(LinearLayout.VERTICAL);
			backLayout.setLayoutParams(new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT));
			backLayout.setGravity(Gravity.BOTTOM);

			// BacktoMainMenu button of the ThankYou screen
			Button backToMainMenu = new Button(this);
			backToMainMenu.setText(getResources().getString(R.string.quest_backtoMainMenu));
			backToMainMenu.setOnClickListener(new View.OnClickListener() {
				public void onClick(View v) {
					finish();
				}
			});
			backLayout.addView(backToMainMenu);
			outerLayout.addView(backLayout);
			setContentView(outerLayout);

			// Puts answers of completed questionnaire together with time stamp and name of questionnaire 
			// in bundle and sends it to QuestionnaireService 
			if (state.answers.length() > 0) {
				JSONObject newAnswerObject = new JSONObject();
				try {
					newAnswerObject.put("timestamp", System.currentTimeMillis());
					newAnswerObject.put("questionnaire", state.questionnaireName);
					newAnswerObject.put("answers", state.answers);
				} catch (JSONException e) {
					e.printStackTrace();
				}
				state.answers = new JSONArray();

				Message msg = Message.obtain();
				Bundle bundle = new Bundle();
				bundle.putString("ACTION", "newAnsweredQuestionnaire");
				bundle.putString("newAnswerObject", newAnswerObject.toString());
				msg.setData(bundle);
				try {
					messenger.send(msg);
				} catch (RemoteException e) {
					e.printStackTrace();
				}
			}
			return;
		}

		// Common to all question types
		Element element = (Element) state.node;

		TextView question = new TextView(this);
		String questionText = element.getElementsByTagName("question").item(0).getTextContent().trim();
		question.setText(questionText);
		question.setTextSize(18);
		question.setTextColor(Color.WHITE);
		question.setPadding(10, 10, 10, 10);
		outerLayout.addView(question);

		ScrollView sLayout = new ScrollView(this);
		sLayout.setLayoutParams(new LayoutParams(LayoutParams.MATCH_PARENT, 0, 1));

		LinearLayout lLayout = new LinearLayout(this);
		lLayout.setOrientation(LinearLayout.VERTICAL);
		lLayout.setLayoutParams(new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT));

		// Question type text: also able to show image
		if (element.getNodeName().equals("text")) {
			NodeList images = element.getElementsByTagName("image");
			if (images.getLength() > 0) {
				String imageName = images.item(0).getTextContent().trim();
				int imageInt = getResources().getIdentifier(imageName, "drawable", getPackageName());

				ImageView image = new ImageView(this);
				image.setLayoutParams(new LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT));
				image.setScaleType(ImageView.ScaleType.CENTER_CROP);
				image.setImageResource(imageInt);
				lLayout.addView(image);
			}	
			
		// Question type edittext: enter free formulated answer	
		} else	if (element.getNodeName().equals("edittext")) {
			TextView prompt = new TextView(this);
			String promptText = element.getElementsByTagName("prompt").item(0).getTextContent().trim();
			prompt.setText(promptText);
			prompt.setTextSize(14);
			prompt.setPadding(10, 0, 10, 5);
			lLayout.addView(prompt);

			LinearLayout vLayout = new LinearLayout(this);
			vLayout.setOrientation(LinearLayout.HORIZONTAL);
			vLayout.setLayoutParams(new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT));
			vLayout.setGravity(Gravity.CENTER_VERTICAL);

			EditText editText = new EditText(this);
			editText.setId(668);
			// Sets number of lines for edit text field defined in questionnaire
			editText.setLines(Integer.parseInt(element.getElementsByTagName("numlines").item(0).getTextContent().trim()));
			editText.setGravity(Gravity.TOP);
			editText.setLayoutParams(new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT, 1));
			editText.setImeOptions(EditorInfo.IME_ACTION_DONE);

			// Listens for changed text answer and saves it
			editText.addTextChangedListener(new TextWatcher() {
				public void afterTextChanged(Editable s) {
				}
				public void beforeTextChanged(CharSequence s, int start, int count, int after) {
				}
				public void onTextChanged(CharSequence s, int start, int before, int count) {
					state.currentAnswer = s.toString();
					Button button = (Button)findViewById(666);
					button.setEnabled(true);
				}
			}); 
			vLayout.addView(editText);

			// Done button hides keyboard
			Button doneButton = new Button(this);
			doneButton.setText(R.string.done);
			doneButton.setOnClickListener(new View.OnClickListener() {
				@Override
				public void onClick(View v) {
					InputMethodManager in = (InputMethodManager) getSystemService(Context.INPUT_METHOD_SERVICE);
					in.hideSoftInputFromWindow(v.getApplicationWindowToken(), InputMethodManager.HIDE_NOT_ALWAYS);
				}
			});
			vLayout.addView(doneButton);

			lLayout.addView(vLayout);
		
		// Question type radio: radio buttons for answer options	
		} else if (element.getNodeName().equals("radio")) {     	       	
			RadioGroup radioGroup = new RadioGroup(this);    	
			NodeList answers = element.getElementsByTagName("choice");

			for (int i=0; i<answers.getLength(); i++ ) {      		
				RadioButton radioButtonText= new RadioButton(this);
				radioButtonText.setText(answers.item(i).getTextContent().trim());
				// Saves chosen radio button's answer option
				radioButtonText.setOnClickListener(new View.OnClickListener(){ 	
					public void onClick(View v) { 
						RadioButton rb = (RadioButton) v;
						state.currentAnswer = rb.getText().toString();

						Button button = (Button)findViewById(666);
						button.setEnabled(true);
					}
				});   
				radioGroup.setPadding(0, 0, 0, 0);	
				radioGroup.addView(radioButtonText);
				if (state.currentAnswer.equals(answers.item(i).getTextContent().trim())) {
					radioButtonText.setChecked(true);
				}
			}
			lLayout.addView(radioGroup);
			
		// Question type check: check boxes as answer options	
		} else if (element.getNodeName().equals("check")) {     	       	   	
			NodeList answers = element.getElementsByTagName("choice");
			state.checkBoxes = new CheckBox[answers.getLength()];
			state.checkBoxCount = 0;

			JSONArray preCheckedList;
			try {
				preCheckedList = new JSONArray(state.currentAnswer);
			} catch (JSONException e) {
				preCheckedList = new JSONArray();
				e.printStackTrace();
			}

			for (int i=0; i<answers.getLength(); i++ ) {      		
				state.checkBoxes[i] = new CheckBox(this);
				state.checkBoxes[i].setText(answers.item(i).getTextContent().trim());
				// Increases checkBoxCount which enables or disables next button
				for (int k=0; k<preCheckedList.length(); k++) {
					if (preCheckedList.optString(k).equals(answers.item(i).getTextContent().trim())) {
						state.checkBoxes[i].setChecked(true);
						state.checkBoxCount = state.checkBoxCount+1;
					}
				}
				lLayout.addView(state.checkBoxes[i]);

				state.checkBoxes[i].setOnClickListener(new View.OnClickListener() {	
					public void onClick(View v) {
						if (((CheckBox) v).isChecked()) {
							state.checkBoxCount = state.checkBoxCount+1;

							// Saves checked boxes answer options
							JSONArray checkedList = new JSONArray();
							for ( int i=0; i<state.checkBoxes.length; i++ ) {
								if (state.checkBoxes[i].isChecked()) {
									checkedList.put(state.checkBoxes[i].getText().toString());
								}
							}
							state.currentAnswer = checkedList.toString();

							Button button = (Button)findViewById(666);
							button.setEnabled(true);
						// No checked box; disables next button 	
						} else {
							state.checkBoxCount = state.checkBoxCount-1;
							if (state.checkBoxCount == 0) {
								Button button = (Button)findViewById(666);
								button.setEnabled(false);
							}
						}
					}
				});        		  		
			}
			
		// Question type likert: likert scale buttons as answer option	
		} else if (element.getNodeName().equals("likert")) {     	       	   	     	
			NodeList answers = element.getElementsByTagName("choice");

			LinearLayout buttonLayout = new LinearLayout(this);
			buttonLayout.setOrientation(LinearLayout.VERTICAL);
			buttonLayout.setLayoutParams(new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT));
			buttonLayout.setGravity(Gravity.CENTER_HORIZONTAL);

			for (int i=0; i<answers.getLength(); i++ ) {      		
				ToggleButton buttonText = new ToggleButton(this);
				buttonText.setText(answers.item(i).getTextContent().trim());
				buttonText.setTextOff(answers.item(i).getTextContent().trim());
				buttonText.setTextOn(answers.item(i).getTextContent().trim());
				buttonText.setLayoutParams(new LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT));
				buttonText.setMinWidth(400);
				buttonLayout.addView(buttonText);

				// Saves clicked answer option 
				buttonText.setOnClickListener(new View.OnClickListener() {
					public void onClick(View v) {
						LinearLayout buttonLayout = (LinearLayout)v.getParent();
						for (int i = 0; i<buttonLayout.getChildCount(); i++) {
							View view = buttonLayout.getChildAt(i);
							if (view instanceof ToggleButton && view!=v) {
								((ToggleButton) view).setChecked(false);
							}
						}
						Button bt = (Button) v;
						state.currentAnswer = bt.getText().toString();

						Button button = (Button)findViewById(666);
						button.setEnabled(true);	
					}
				});

				if (state.currentAnswer.equals(answers.item(i).getTextContent().trim())) {
					buttonText.setChecked(true);
				}
			}
			lLayout.addView(buttonLayout);
			
		// Question type dropdown: dropdown menu for answer options	
		} else if(element.getNodeName().equals("dropdown")) {
			NodeList answers = element.getElementsByTagName("choice");

			Spinner spinner = new Spinner(this);
			ArrayAdapter<String> spinnerArrayAdapter = new ArrayAdapter<String>(this, android.R.layout.simple_dropdown_item_1line, new ArrayList<String>());
			spinner.setAdapter(spinnerArrayAdapter);
			spinnerArrayAdapter.add(getResources().getString(R.string.quest_spinnerPleaseChoose));
			for (int i=0; i<answers.getLength(); i++ ) {  
				spinnerArrayAdapter.add(answers.item(i).getTextContent().trim());
				if (state.currentAnswer.equals(answers.item(i).getTextContent().trim())) {
					spinner.setSelection(i+1);
				}
			}
			lLayout.addView(spinner);
			
			// Saves chosen answer option and enables / disables next button
			spinner.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {	
				public void onItemSelected (AdapterView<?> parent, View v, int position, long id) {
					if (position == 0) {
						state.currentAnswer = "";
						((Button)findViewById(666)).setEnabled(false);
					} else {
						Spinner sp = (Spinner) parent;
						state.currentAnswer = (String) sp.getSelectedItem();

						((Button)findViewById(666)).setEnabled(true);
					}
				}
				public void onNothingSelected (AdapterView<?> parent) {
				}
			});
			
		// Question type scaleedit: bar with slider and edittext to enter value	
		} else if (element.getNodeName().equals("scaleedit")) {   
			LinearLayout seekBarLayout = new LinearLayout(this);
			seekBarLayout.setOrientation(LinearLayout.VERTICAL);
			seekBarLayout.setLayoutParams(new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT));
			seekBarLayout.setPadding(40, 0, 40, 0);

			LinearLayout scaleLayout = new LinearLayout(this);
			scaleLayout.setOrientation(LinearLayout.HORIZONTAL);
			scaleLayout.setLayoutParams(new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT));
			scaleLayout.setGravity(Gravity.RIGHT);
			scaleLayout.setPadding(25, 0, 25, 0);
			String maxValue = (element.getElementsByTagName("maxValue").item(0).getTextContent().trim());

			// Sets value in edit text field after changed slider position
			final SeekBar seekBar = new SeekBar(this);
			String seekValue = element.getElementsByTagName("value").item(0).getTextContent().trim();
			seekBar.setMax(Integer.parseInt(maxValue));
			seekBar.setProgress(Integer.valueOf(seekValue));
			seekBar.setOnSeekBarChangeListener(new  OnSeekBarChangeListener() {
				@Override
				public void onProgressChanged(SeekBar seekBar, int progress,
						boolean fromUser) {
					EditText editValue = (EditText)findViewById(667);
					editValue.setText(String.valueOf(progress));					
				}
				@Override
				public void onStartTrackingTouch(SeekBar seekBar) {
				}
				@Override
				public void onStopTrackingTouch(SeekBar seekBar) {
				}
			});
			seekBarLayout.addView(seekBar);

			// Sets value of slider after changed edit text value 
			final EditText editValue = new EditText(this);
			editValue.setText(seekValue);
			editValue.setId(667);
			editValue.setGravity(Gravity.RIGHT);
			editValue.setInputType(InputType.TYPE_CLASS_NUMBER);
			editValue.setMaxWidth(150);
			editValue.setFilters(new InputFilter[]{ new InputFilterMinMax("0", maxValue)});
			editValue.setMinWidth(90);

			editValue.addTextChangedListener(new TextWatcher() {
				@Override
				public void afterTextChanged(Editable s) {
				}
				@Override
				public void beforeTextChanged(CharSequence s, int start, int count, int after) {
				}
				@Override
				public void onTextChanged(CharSequence s, int start, int before, int count) {
					int current_value;
					try {
						current_value = Integer.parseInt(editValue.getText().toString());
						editValue.setSelection(count);

					} catch (Throwable t) {
						current_value = 0;
					}
					seekBar.setProgress(current_value);
					state.currentAnswer = String.valueOf(current_value);
				}
			}); 
			scaleLayout.addView(editValue);

			TextView unit = new TextView(this);
			String unitText = element.getElementsByTagName("unit").item(0).getTextContent().trim();
			unit.setText("  " + unitText);
			unit.setTextSize(15);
			unit.setTextColor(Color.WHITE);
			scaleLayout.addView(unit); 
			lLayout.addView(seekBarLayout);
			lLayout.addView(scaleLayout);

			state.currentAnswer = seekValue;
		}
		RelativeLayout rLayout = new RelativeLayout(this);
		rLayout.setLayoutParams(new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT));

		// Next button to switch to next question
		Button nextButton = new Button(this);
		nextButton.setId(666);
		nextButton.setText(getResources().getString(R.string.quest_nextQuestion));
		nextButton.setLayoutParams(new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT));
		nextButton.setEnabled(false);
		nextButton.setCompoundDrawablesWithIntrinsicBounds(0, 0, R.drawable.ic_arrow, 0);

		// Saves current chosen answer in answers list of questionnaire, deletes current answer, 
		// skips to next question
		nextButton.setOnClickListener(new View.OnClickListener(){ 	
			public void onClick(View v) {
				state.answers.put(state.currentAnswer);
				state.currentAnswer = "";

				state.node = state.node.getNextSibling();
				while (state.node != null && state.node.getNodeName().equals("#text")) {
					state.node = state.node.getNextSibling();
				}
				InputMethodManager in = (InputMethodManager) getSystemService(Context.INPUT_METHOD_SERVICE);
				in.hideSoftInputFromWindow(v.getApplicationWindowToken(), InputMethodManager.HIDE_NOT_ALWAYS);
				displayCurrentQuestion();
			}
		});

		sLayout.addView(lLayout);
		outerLayout.addView(sLayout);

		rLayout.addView(nextButton);
		rLayout.setGravity(Gravity.BOTTOM);	
		outerLayout.addView(rLayout);

		setContentView(outerLayout);

		// Next button for question type text scale edit always enabled
		if (element.getNodeName().equals("text") || element.getNodeName().equals("scaleedit") || state.currentAnswer.length()>0) {
			nextButton.setEnabled(true);
		}
	}

	// Input filter for scale edit's edit text field, allows no smaller values than minimum value and
	// no higher values than maximum value of seek bar
	public class InputFilterMinMax implements InputFilter {
		private int min, max;

		public InputFilterMinMax(int min, int max) {
			this.min = min;
			this.max = max;
		}

		public InputFilterMinMax(String min, String max) {
			this.min = Integer.parseInt(min);
			this.max = Integer.parseInt(max);
		}

		@Override
		public CharSequence filter(CharSequence source, int start, int end, Spanned dest, int dstart, int dend) {	
			try {
				String startString = dest.toString().substring(0, dstart);
				String insert = source.toString();
				String endString = dest.toString().substring(dend);
				String parseThis = startString+insert+endString;
				int input = Integer.parseInt(parseThis);
				if (isInRange(min, max, input))
					return null;
			} catch (NumberFormatException nfe) {
			}		
			return "";
		}
		private boolean isInRange(int a, int b, int c) {
			return b > a ? c >= a && c <= b : c >= b && c <= a;
		}
	}

	// Loads chosen questionnaire and finds first question of questionnaire
	public void loadQuestionnaire(String q) {
		Document doc = null;
		try{
			InputSource is = new InputSource(new StringReader(q));
			DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
			DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
			doc = docBuilder.parse(is);
			doc.getDocumentElement().normalize();
		} catch (Exception e) {
			e.printStackTrace();
		}

		state.questionnaireName = doc.getElementsByTagName("title").item(0).getTextContent().trim();
		state.node = doc.getElementsByTagName("questions").item(0).getFirstChild();
		while (state.node.getNodeName().equals("#text")){
			state.node = state.node.getNextSibling();
		}
	}

	// Shows dialog window in case back button is pushed
	@Override
	protected Dialog onCreateDialog(int id) {
		final Dialog dialog;
		switch(id) {
		case DIALOG_BACK_ALERT_ID:
			dialog = new AlertDialog.Builder(this).setTitle(getResources().getString(R.string.backAlertTitle))
			.setMessage(getResources().getString(R.string.backAlertDialog))
			.setCancelable(true)
			.setPositiveButton(getResources().getString(R.string.ok), new DialogInterface.OnClickListener() {
				public void onClick(DialogInterface dialog, int id) {
					QuestionnaireAsker.this.finish();
				}
			})
			.setNegativeButton(getResources().getString(R.string.cancel), new DialogInterface.OnClickListener() {
				public void onClick(DialogInterface dialog, int id) {
					dialog.cancel();
				}
			}).create();
			break;
		default:
			dialog = null;
		}
		return dialog;
	}

	// Receives pushed back button 
	@Override
	public boolean onKeyDown(int keyCode, KeyEvent event) {
		if (keyCode == KeyEvent.KEYCODE_BACK)
			showDialog(DIALOG_BACK_ALERT_ID);

		return true;
	}
}
