/*******************************************************************************
 * Copyright (c) 2012 Barbara Schmid.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Barbara Schmid - initial API and implementation
 ******************************************************************************/
package com.questionnaire;

import java.util.Iterator;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.ComponentName;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.ServiceConnection;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.os.IBinder;
import android.os.Message;
import android.os.Messenger;
import android.os.RemoteException;
import android.util.Log;
import android.view.Display;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.widget.Button;
import android.widget.LinearLayout;
import android.widget.LinearLayout.LayoutParams;
import android.widget.ScrollView;
import android.widget.TextView;

public class QuestionnaireChooser extends Activity {
	
	// Upload answers button of options menu
	private static final int UPLOAD_ANSWERS_ID=Menu.FIRST+1;
	
	// Settings button of options menu
	private static final int SETTINGS_ID=Menu.FIRST+2;
	
	// Messenger for communication with QuestionnaireService
	private Messenger messenger = null;

	@Override
	public boolean onPrepareOptionsMenu(Menu menu) {
		super.onPrepareOptionsMenu(menu);
		return true;
	}

	@Override
	public boolean onCreateOptionsMenu(Menu menu) {
		super.onCreateOptionsMenu(menu);
		menu.add(0, UPLOAD_ANSWERS_ID, 0, getResources().getString(R.string.pref_menu_upload)).setIcon(R.drawable.ic_upload);
		menu.add(0, SETTINGS_ID, 1, getResources().getString(R.string.pref_menu_settings)).setIcon(R.drawable.ic_settings);
		return true;
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item) {
		switch (item.getItemId()) {
		// Upload answers button pushed, sends QuestionnaireService task to upload answers
		case UPLOAD_ANSWERS_ID:
			if (messenger != null) {
				Message msg = Message.obtain();
				msg.replyTo = new Messenger(handler);
				Bundle bundle = new Bundle();
				bundle.putString("ACTION", "triggerUpload");
				msg.setData(bundle);

				try {
					messenger.send(msg);
				} catch (RemoteException e) {
					e.printStackTrace();
				}
			}
			return true;
		// Settings button pushed, starts EditPreferences	
		case SETTINGS_ID:
			startActivity (new Intent (this, EditPreferences.class));
			return true;
		default:
			return super.onOptionsItemSelected(item);
		}
	}

	// Requests available questionnaires from QuestionnaireService
	private ServiceConnection conn = new ServiceConnection() {
		@Override
		public void onServiceConnected(ComponentName className, IBinder binder) {
			messenger = new Messenger(binder);

			Message msg = Message.obtain();
			msg.replyTo = new Messenger(handler);
			Bundle bundle = new Bundle();
			bundle.putString("ACTION", "getQuestionnaire");
			msg.setData(bundle);

			try {
				messenger.send(msg);
			} catch (RemoteException e) {
				e.printStackTrace();
			}
		}
		@Override
		public void onServiceDisconnected(ComponentName className) {
			messenger = null;
		}
	};

	private Handler handler = new Handler () {
		@Override
		public void handleMessage(Message message) {

			ScrollView sLayout = new ScrollView(QuestionnaireChooser.this);
			sLayout.setLayoutParams(new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT));

			Bundle data = message.getData();
			if (message.arg1 == RESULT_OK && data != null) {
				String monitoredPackagenames = data.getString("monitoredPackagenames");
				
				// Show welcome screen, includes task for study and monitored package names
				if (monitoredPackagenames != null) {
					String welcomeTitle = getResources().getString(R.string.welcomeTitle);
					String welcomeText = String.format(getResources().getString(R.string.welcomeText), monitoredPackagenames);
					new AlertDialog.Builder(QuestionnaireChooser.this).setIcon(android.R.drawable.ic_dialog_alert).setTitle(welcomeTitle).setMessage(welcomeText).setPositiveButton(
							R.string.ok, new DialogInterface.OnClickListener() {
								public void onClick(DialogInterface dialog, int which) {
									dialog.dismiss();
								}
							}).show();
				}

				// Shows list of buttons with names of missing package names
				String missingJSON = data.getString("missing");
				JSONArray missing;
				if (missingJSON != null) {
					try {
						missing = new JSONArray(missingJSON);
					} catch (JSONException e) {
						missing = new JSONArray();
						e.printStackTrace();
					}

					LinearLayout lLayout = new LinearLayout(QuestionnaireChooser.this);				
					lLayout.setOrientation(LinearLayout.VERTICAL);
					lLayout.setLayoutParams(new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT));
					lLayout.setPadding(10, 20, 10, 10);

					TextView tv = new TextView(QuestionnaireChooser.this);
					tv.setText(getResources().getString(R.string.chooser_appIsMissing));
					tv.setPadding(25, 25, 25, 25);
					lLayout.addView(tv);


					for (int i=0; i<missing.length(); i++) {
						final String pname = missing.optString(i);

						Button b = new Button(QuestionnaireChooser.this);
						b.setText(String.format(getResources().getString(R.string.chooser_appStoreButton), pname));
						// Button with missing app clicked, opens either Google play app or browser 
						// for page of clicked missing app
						b.setOnClickListener(new View.OnClickListener(){
							@Override
							public void onClick(View v) {
								try {
									Intent market = new Intent(Intent.ACTION_VIEW, Uri.parse("market://details?id="+pname));
									market.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
									startActivity(market);
								} catch (android.content.ActivityNotFoundException anfe) {
									Intent browser = new Intent(Intent.ACTION_VIEW, Uri.parse("http://play.google.com/store/apps/details?id="+pname));
									browser.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
									startActivity(browser);
								}
							}
						});
						LinearLayout.LayoutParams layoutParams = new LinearLayout.LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT);
						layoutParams.setMargins(24, 50, 24, 0);
						lLayout.addView(b,layoutParams);
					}
					sLayout.addView(lLayout);
					setContentView(sLayout);

				// Show list of available questionnaires as buttons	
				} else {
					String questionnairesJSON = data.getString("questionnaires");
					JSONObject questionnaires = null;
					try {
						questionnaires = new JSONObject(questionnairesJSON);
					} catch (JSONException e) {
						e.printStackTrace();
					}

					LinearLayout lLayout = new LinearLayout(QuestionnaireChooser.this);				
					lLayout.setOrientation(LinearLayout.VERTICAL);
					lLayout.setLayoutParams(new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT));
					lLayout.setPadding(10, 20, 10, 10);

					for (Iterator<?> iter = questionnaires.keys(); iter.hasNext(); ) {
						String title = (String) iter.next();

						Display display = getWindowManager().getDefaultDisplay();
						int screenHeight = display.getHeight();

						Button questionnaire_name_Button = new Button(QuestionnaireChooser.this);
						questionnaire_name_Button.setText(title);
						questionnaire_name_Button.setLayoutParams(new LinearLayout.LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT));
						questionnaire_name_Button.setHeight(screenHeight/6);

						final String ftitle = title;
						try {
							final String xml = questionnaires.getString(title);
							// Start QuestionnaireAsker, send information about chosen questionnaire to it
							questionnaire_name_Button.setOnClickListener(new View.OnClickListener(){
								@Override
								public void onClick(View v) { 
									Intent intentMain = new Intent(QuestionnaireChooser.this , QuestionnaireAsker.class);
									intentMain.putExtra("title", ftitle);
									intentMain.putExtra("xml", xml);
									QuestionnaireChooser.this.startActivity(intentMain);
								}
							});
						} catch (JSONException e) {
							e.printStackTrace();
						}					
						lLayout.addView(questionnaire_name_Button);
					}
					sLayout.addView(lLayout);
					setContentView(sLayout);
				}
			}
		}
	};

	@Override
	public void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		Log.d("Chooser", "onCreate");
	}

	@Override
	protected void onResume() {
		super.onResume();
		Log.d("Chooser", "onResume");

		Intent intent = new Intent(getApplicationContext(), QuestionnaireService.class);
		bindService(intent, conn, 0);
	}

	@Override
	protected void onPause() {
		unbindService(conn);
		Log.d("Chooser", "onPause");
		super.onPause();
	}
}
