/*******************************************************************************
 * Copyright (c) 2012 Barbara Schmid.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Barbara Schmid - initial API and implementation
 *     Andreas Möller - comments
******************************************************************************/
package com.questionnaire;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringReader;
import java.math.BigInteger;
import java.security.SecureRandom;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.params.BasicHttpParams;
import org.apache.http.params.HttpParams;
import org.apache.http.protocol.HTTP;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;

import android.app.Activity;
import android.app.ActivityManager;
import android.app.AlarmManager;
import android.app.KeyguardManager;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.app.Service;
import android.content.BroadcastReceiver;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.SharedPreferences;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.content.res.AssetManager;
import android.location.Criteria;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Bundle;
import android.os.Environment;
import android.os.Handler;
import android.os.IBinder;
import android.os.Message;
import android.os.Messenger;
import android.preference.PreferenceManager;
import android.util.Log;
import android.view.Display;
import android.view.WindowManager;



public class QuestionnaireService extends Service {

	/* %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
	   %%%%%%%%%%%%%%  PARAMETER SECTION  %%%%%%%%%%%%%%%%%%%%%%%%%
	   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% */   

	// String for server IP
	private String serverIP = "ENTER YOUR SERVER IP/PORT HERE";
	
	// Interval in seconds for manual upload reminder notification 
	private long murTime = 60*60*24;
	
	// Interval in seconds for periodic download from backend server
	private long pdTime = 60*60*12;
	
	// Interval in seconds for saving logs
	private long slTime = 60*60;


	// Uses questionnaires from cache if true and questionnaires from assets folder if false 
	private Boolean isReset = false;
	
	// Group ID for study groups
	private String groupID = "1";
	
	// Questionnaire mode
	private Mode mode = Mode.VOLUNTARY;

	// List of all questionnaire modes
	private enum Mode {
		VOLUNTARY, INTERVALTRIGGERED, EVENTTRIGGERED
	};
	
	// %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

	// Currently active questionnaires
	private JSONObject questionnaires;
	// Questionnaires that are not activated yet
	private JSONObject updatedQuestionnaires;
	// Answered questionnaires
	private JSONArray answeredQuestionnaires;
	// Saves number of times questionnaire has been answered
	private JSONObject answerCount;

	// Logged data
	private JSONArray logArray;
	// List with monitored activities of the apps
	private List<String> monitoredActivities = new ArrayList<String>();
	// List with monitored package names of the apps
	private List<String> monitoredPackagenames = new ArrayList<String>();
	// Maps package name to corresponding questionnaire
	private JSONObject packagenameToQuestionnaireMap = new JSONObject();
	// Variable to track last usage of app
	private JSONObject lastUsed = new JSONObject();
	// Variable to track whether screen is off
	private boolean screenOff;
	// Variable for GPS location
	private Location location;
	// Logger
	Thread loggerThread;

	// Pending intent for manual upload reminder notification
	private PendingIntent piMUR;
	// Pending intent for periodic download of questionnaires from server
	private PendingIntent piPD;
	// Pending intent for questionnaire moving
	private PendingIntent piQM;
	// Pending intent for saving logged data
	private PendingIntent piSL;

	// Messenger for communication with QuestionnaireChooser and QuestionnaireAsker
	private Messenger chooserMessenger = null;
	private final Messenger inMessenger = new Messenger(new IncomingHandler());

	// User ID for each smartphone
	private String userId;

	// Logger
	private class LoggerRunnable implements Runnable {
		@Override
		public void run() {
			ActivityManager am = (ActivityManager) getSystemService(ACTIVITY_SERVICE);
			KeyguardManager kgMgr = (KeyguardManager) getSystemService(Context.KEYGUARD_SERVICE);

			String previousActivity = new String();
			String previousPackagename = new String();
			long previousTimestamp = 0;
			boolean previousIsMonitored = false;

			// First log entry after logger is started
			try {
				JSONObject logObject = new JSONObject();
				logObject.put("userId", userId);
				logObject.put("timestamp", System.currentTimeMillis());
				logObject.put("duration", 0);		
				logObject.put("activity", "com.questionnaire.Logger");
				logObject.put("packagename", "com.questionnaire");
				logObject.put("topackagename", "");
				if (location != null) {
					logObject.put("location", Double.toString(location.getAltitude())+","+
							Double.toString(location.getLongitude())+","+
							Double.toString(location.getLatitude()));
				} else {
					logObject.put("location", "provider is deactivated");
				}

				synchronized(logArray) {
					logArray.put(logObject);
				}
			} catch (JSONException e) {
				e.printStackTrace();
			}

			while(true) {
				String activity;
				String packagename;
				long timestamp = System.currentTimeMillis();

				// Logs if smartphone sleeps
				if (screenOff == true) {	
					activity = "Sleeping";
					packagename = "sleeping";
				}
				// Logs if smartphone is locked
				else if (kgMgr.inKeyguardRestrictedInputMode()) {
					activity = "Locked";
					packagename = "locked";
				}
				else{
					List<ActivityManager.RunningTaskInfo > taskInfo = am.getRunningTasks(1);
					ComponentName componentInfo = taskInfo.get(0).topActivity;
					componentInfo.getPackageName();

					activity = taskInfo.get(0).topActivity.getClassName();
					packagename = componentInfo.getPackageName();
				}

				// Use these two lines to gather activity and package name of an app to be logged
				//Log.d("LOGActivity:", activity);
				//Log.d("LOGPackageName:", packagename);

				// Logs activity in case it has to be logged
				if (previousActivity.equals(activity) == false) {
					if (previousIsMonitored == true) {
						try {
							JSONObject logObject = new JSONObject();
							logObject.put("userId", userId);
							logObject.put("timestamp", previousTimestamp);
							logObject.put("duration", timestamp-previousTimestamp);		
							logObject.put("activity", previousActivity);
							logObject.put("packagename", previousPackagename);
							logObject.put("topackagename", packagename);
							if (location != null) {
								logObject.put("location", Double.toString(location.getAltitude())+","+
										Double.toString(location.getLongitude())+","+
										Double.toString(location.getLatitude()));
							} else {
								logObject.put("location", "provider is deactivated");
							}

							synchronized(logArray) {
								logArray.put(logObject);
								String pname = packagenameToQuestionnaireMap.optString(previousPackagename);
								if (pname.length()>0) {
									lastUsed.put(questionnaireLogPackage(questionnaires.optString(pname)), timestamp);
								}
							}
						} catch (JSONException e) {
							e.printStackTrace();
						}

						// Monitored application has been left, autoopen the questionnaire if necessary
						if (previousPackagename.equals("com.questionnaire") == false) {
							if (packagename.equals("com.sec.android.app.twlauncher") ||
									packagename.equals("com.android.launcher") ||
									packagename.equals("com.sec.android.app.launcher") ||
									packagename.equals("com.htc.launcher") ||
									packagename.equals("com.gau.go.launcherex") ||
									packagename.equals("com.cyanogenmod.trebuchet") ||
									packagename.equals("com.anddoes.launcher") ||
									packagename.equals("com.teslacoilsw.launcher") ||
									packagename.equals("org.adw.launcher")){
								String title = packagenameToQuestionnaireMap.optString(previousPackagename);
								if (title.length()>0 && questionnaireAutoOpen(questionnaires.optString(title)) == true) {
									Intent intentMain = new Intent(getBaseContext(), QuestionnaireAsker.class);
									intentMain.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
									intentMain.putExtra("title", title);
									intentMain.putExtra("xml", questionnaires.optString(title));
									getApplication().startActivity(intentMain);
								}
							}
						}
					}

					previousActivity = activity;
					previousPackagename = packagename;
					previousTimestamp = timestamp;

					previousIsMonitored = monitoredActivities.contains(previousActivity) == true 
							|| monitoredPackagenames.contains(previousPackagename) == true 
							|| previousPackagename.equals("com.questionnaire");
				}
				try {
					Thread.currentThread();
					Thread.sleep(1000);
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
		}
	}

	// Checks screen state of smartphone
	private class ScreenReceiver extends BroadcastReceiver {
		@Override
		public void onReceive(Context context, Intent intent) {
			if (intent.getAction().equals(Intent.ACTION_SCREEN_OFF)) {
				screenOff = true;
			} else if (intent.getAction().equals(Intent.ACTION_SCREEN_ON)) {
				screenOff = false;
			}
		}
	}
	
	// Listens for GPS location updates
	private LocationListener locationListener = new LocationListener() {
		public void onLocationChanged(Location loc) {
			location = loc;
		}
		public void onStatusChanged(String provider, int status, Bundle extras) {}

		public void onProviderEnabled(String provider) {}

		public void onProviderDisabled(String provider) {}
	};


	// Update monitored activities and package names for active questionnaires
	private void updateMonitored() {
		monitoredActivities.clear();
		monitoredPackagenames.clear();
		packagenameToQuestionnaireMap = new JSONObject();

		PackageManager pm = getPackageManager();
		List<ApplicationInfo> packages = pm.getInstalledApplications(PackageManager.GET_META_DATA);

		synchronized(questionnaires) {
			for (Iterator<?> iter = questionnaires.keys(); iter.hasNext(); ) {
				String title = (String) iter.next();
				String xml;
				try {
					xml = questionnaires.getString(title);
					String logActivity = questionnaireLogActivity(xml);
					String acts[] = logActivity.split(",");
					for (int i=0; i<acts.length; i++) {
						if (acts[i].trim().length()>0 && monitoredActivities.contains(acts[i].trim()) == false) {
							monitoredActivities.add(acts[i].trim());
						}
					}
					String logPackage = questionnaireLogPackage(xml);
					String pacs[] = logPackage.split(",");
					for (int i=0; i<pacs.length; i++) {
						// Goes through all regexes
						if (pacs[i].trim().length()>0) {
							// Goes through all installed packages
							for (ApplicationInfo packageInfo : packages) {
								if (packageInfo.packageName.matches(pacs[i].trim())) {
									// Adds only package names that are not already in list
									if (monitoredPackagenames.contains(packageInfo.packageName) == false) {
										monitoredPackagenames.add(packageInfo.packageName);
										// For autoopen of questionnaires, create mapping from package name to corresponding questionnaire
										if (mode != Mode.EVENTTRIGGERED || questionnaireAutoOpen(xml) == true) {
											packagenameToQuestionnaireMap.put(packageInfo.packageName, title);        
										}
									}
								}
							}
						}
					}
				} catch (JSONException e) {
					e.printStackTrace();
				}
			}
		}
	}

	// Creates list of missing apps or list of active questionnaires for QuestionnaireChooser
	private Bundle createChooserData() {
		Bundle bundle = new Bundle();
		synchronized(questionnaires) {
			if (questionnaires.length() == 0) return null;

			PackageManager pm = getPackageManager();
			List<ApplicationInfo> packages = pm.getInstalledApplications(PackageManager.GET_META_DATA);

			// Creates list of missing apps
			JSONArray missing = new JSONArray();
			for (Iterator<?> iter = questionnaires.keys(); iter.hasNext(); ) {
				String title = (String) iter.next();
				String xml = questionnaires.optString(title);
				String logPackage = questionnaireLogPackage(xml);
				String pacs[] = logPackage.split(",");
				for (int i=0; i<pacs.length; i++) {
					boolean found = false; 
					// Goes through all regexes
					if (pacs[i].trim().length()>0) {
						// Goes though all installed packages and find matches
						for (ApplicationInfo packageInfo : packages) {
							if (packageInfo.packageName.matches(pacs[i].trim())) {
								found = true;
								break;
							}
						}
						// Package not installed, add to list of missing apps 
						if (found == false) {
							missing.put(pacs[i].trim());
						}
					}
				}
			}

			// Puts either questionnaires or missing apps in bundle
			if (missing.length() == 0) {
				bundle.putString("questionnaires", questionnaires.toString());
			} else {
				bundle.putString("missing", missing.toString());
			}
		}

		SharedPreferences mPrefs = getSharedPreferences("QuestionnairePreferences", MODE_PRIVATE);
		Boolean welcomeScreenShown = mPrefs.getBoolean("welcomeScreenShown", false);
		// Creates data for welcome screen of QuestionnaireChooser 
		if (welcomeScreenShown == false) {
			bundle.putString("monitoredPackagenames", concatStringsWSep(monitoredPackagenames, "\n"));
			SharedPreferences.Editor editor = mPrefs.edit();
			editor.putBoolean("welcomeScreenShown", true);
			editor.commit();
		}
		return bundle;
	}

	// Saves logged data and last usage of app in shared preferences
	private void saveLogObject() {
		synchronized(logArray) {
			SharedPreferences prefs = getSharedPreferences("QuestionnairePreferences", MODE_PRIVATE);  
			SharedPreferences.Editor prefEditor = prefs.edit();
			prefEditor.putString("lastUsed", lastUsed.toString());
			prefEditor.putString("logArray", logArray.toString());
			prefEditor.commit();
		}
	}

	// Uploads logged data to backend server and saves logs on SD card
	public void uploadLog(){
		if (isNetworkAvailable() == false)
			return;

		saveLogObject();
		final JSONArray la;
		
		// Uploads logs to backend server, if connection failure info log message
		synchronized(logArray) {
			try {			
				HttpParams httpParameters = new BasicHttpParams();
				HttpClient client = new DefaultHttpClient(httpParameters);
				HttpPost request = new HttpPost(serverIP + "/logs");

				StringEntity se = new StringEntity(logArray.toString());
				request.setEntity(se);
				request.setHeader("Accept", "application/json");
				request.setHeader("Content-type", "application/json");

				HttpResponse response = client.execute(request);
				if (response.getStatusLine().getStatusCode() == 200) {
					Log.i("Service", "Log file uploaded successfully");
					la = logArray;
					logArray = new JSONArray();
				} else {
					Log.i("Service", "Log file upload failed");
					return;
				}
			} catch (Exception e) {
				Log.i("Service", "Log file upload timed out?");
				return;
			}
		}

		Log.i("Service", "saveLog");
		// Saves logs as txt file on SD card
		new Thread(new Runnable() {
			@Override
			public void run() {
				String state = Environment.getExternalStorageState();
				if (Environment.MEDIA_MOUNTED.equals(state)) {
					Log.i("Service", "Saving log to file");
					File dir = getExternalFilesDir(null);
					File logFile = new File(dir, "Log.txt");
					try {
						FileWriter fw = new FileWriter(logFile, true);
						for (int i = 0; i<la.length(); i++) {
							try {
								fw.append(la.getJSONObject(i).toString() + '\n');
							} catch (JSONException e) {
								e.printStackTrace();
							}
						}
						fw.flush();
					} catch (IOException e) {
						e.printStackTrace();
					}
				} else {
					Log.i("Service", "Could not save log to SD card");
				}
			}
		}).start();
	}

	// Receives the different alarms
	private class AlarmReceiver extends BroadcastReceiver {
		// ID for manual upload reminder notification
		final int UPLOAD_ID = 1;
		// ID for scheduled questionnaire notification
		final int FILLOUT_ID = 2;

		@Override
		public void onReceive(Context context, Intent intent) {
			String action = intent.getAction();
			Log.i("Service", "Alarm " + action);

			// Manual upload reminder notification
			if (action.equals("com.questionnaire.ManualUploadReminder")) {
				SharedPreferences sharedPreferences = PreferenceManager.getDefaultSharedPreferences(QuestionnaireService.this);
				boolean autoUpload = sharedPreferences.getBoolean("pref_auto_upload_title", true);
				if (autoUpload == true) {
					new Thread(new Runnable() {
						@Override
						public void run() {
							uploadAnswers();
						}
					}).start();
				// Automatic upload in settings menu deactivated	
				} else {
					String ns = Context.NOTIFICATION_SERVICE;
					NotificationManager mNotificationManager = (NotificationManager) getSystemService(ns);

					int icon = R.drawable.ic_quest_notification;
					CharSequence tickerText1 = getResources().getString(R.string.mur_tickerText);
					long when = System.currentTimeMillis();

					Notification notification1 = new Notification(icon, tickerText1, when);
					notification1.flags = Notification.DEFAULT_VIBRATE | Notification.FLAG_AUTO_CANCEL;

					Context context1 = getApplicationContext();

					CharSequence contentTitle1 = getResources().getString(R.string.mur_contentTitle);
					CharSequence contentText1 = getResources().getString(R.string.mur_contentText);

					Intent notificationIntent1 = new Intent("com.questionnaire.UploadManual");
					PendingIntent contentIntent1 = PendingIntent.getBroadcast(QuestionnaireService.this, 0, notificationIntent1, 0);

					notification1.setLatestEventInfo(context1, contentTitle1, contentText1, contentIntent1);
					mNotificationManager.notify(UPLOAD_ID, notification1);
				}
				
			// Notification for interval triggered scheduled questionnaire	
			} else if (action.equals("com.questionnaire.ScheduledQuestionnaire")) {
				String title = intent.getExtras().getString("title");
				String packageName = questionnaireLogPackage(questionnaires.optString(title));
				long last = 0;

				// Sets last usage of monitored apps
				String pacs[] = packageName.split(",");
				for (int i=0; i<pacs.length; i++) {
					if (pacs[i].trim().length()>0) {
						last = lastUsed.optLong(pacs[i].trim());
						if (last > 0) break;
					}
				}

				// Shows notification if no questionnaire has been answered after logged app usage
				if (packageName.length()==0 || last>0) {
					String ns = Context.NOTIFICATION_SERVICE;
					NotificationManager mNotificationManager = (NotificationManager) getSystemService(ns);

					int icon = R.drawable.ic_quest_notification;
					CharSequence tickerText2 = getResources().getString(R.string.sq_tickerText);
					long when = System.currentTimeMillis();

					Notification notification2 = new Notification(icon, tickerText2, when);
					notification2.flags = Notification.DEFAULT_VIBRATE | Notification.DEFAULT_SOUND | Notification.FLAG_AUTO_CANCEL;

					Context context2 = getApplicationContext();

					CharSequence contentTitle2 = String.format(getResources().getString(R.string.sq_contentTitle), title);
					CharSequence contentText2 = getResources().getString(R.string.sq_contentText);

					// Starts QuestionnaireAsker if notification has been clicked
					Intent notificationIntent2 = new Intent(QuestionnaireService.this, QuestionnaireAsker.class);
					notificationIntent2.putExtra("title", title);
					String xml = null;
					synchronized(questionnaires) {
						xml = questionnaires.optString(title);
					}

					notificationIntent2.putExtra("xml", xml);
					notificationIntent2.setAction(title);
					PendingIntent contentIntent2 = PendingIntent.getActivity(QuestionnaireService.this, 0, notificationIntent2, PendingIntent.FLAG_CANCEL_CURRENT);
					notification2.setLatestEventInfo(context2, contentTitle2, contentText2, contentIntent2);
					mNotificationManager.notify(title, FILLOUT_ID, notification2);
				}
				setAlarmSQ();
				
			// Triggers periodic download of questionnaires from backend server	
			} else if (action.equals("com.questionnaire.PeriodicDownloader")) {
				new Thread(new Runnable() {
					@Override
					public void run() {
						downloadQuestionnaires();
					}
				}).start();

			// Triggers moveQuestionnaire function	
			} else if (action.equals("com.questionnaire.QuestionnaireMover")) {
				moveQuestionnaires();

			// Triggers upload of answers	
			} else if (action.equals("com.questionnaire.UploadManual")) {
				new Thread(new Runnable() {
					@Override
					public void run() {
						uploadAnswers();
					}
				}).start();
				
			// Triggers function to save logged data	
			} else if (action.equals("com.questionnaire.SaveLog")) {
				new Thread(new Runnable() {
					@Override
					public void run() {
						saveLogObject();
					}
				}).start();
			}
		}
	}

	// Sets ManualUploadReminder alarm in case there are answered questionnaires that have not been sent to backend server
	// after mur time defined in parameter section has passed again
	private void setAlarmMUR() {
		AlarmManager am = (AlarmManager)getSystemService(Context.ALARM_SERVICE);

		long timestamp = Long.MAX_VALUE;
		for (int i= 0; i < answeredQuestionnaires.length(); i++) {
			long ts;
			ts = answeredQuestionnaires.optJSONObject(i).optLong("timestamp");
			if (ts < timestamp) {
				timestamp = ts;
			}
		}

		if (timestamp != Long.MAX_VALUE) {
			SharedPreferences sharedPreferences = PreferenceManager.getDefaultSharedPreferences(QuestionnaireService.this);
			boolean autoUpload = sharedPreferences.getBoolean("pref_auto_upload_title", true);
			if (autoUpload == true) {
				am.set(AlarmManager.RTC_WAKEUP, System.currentTimeMillis()+60*1000, piMUR);
			} else {
				am.set(AlarmManager.RTC_WAKEUP, timestamp + murTime*1000, piMUR);					
			}
		}
	}

	// Sets QuestionnaireMover alarm that activates questionnaires, which start date is reached, 
	// and deactivates questionnaires, which end date is reached
	private void setAlarmQM() {
		AlarmManager am = (AlarmManager)getSystemService(Context.ALARM_SERVICE);		
		long timestamp = Long.MAX_VALUE;

		synchronized(questionnaires) {
			for (Iterator<?> iter = updatedQuestionnaires.keys(); iter.hasNext(); ) {
				String title = (String) iter.next();
				String xml = updatedQuestionnaires.optString(title);
				long ts = questionnaireStartDate(xml);
				if (ts < timestamp) {
					timestamp = ts;
				}
				ts = questionnaireEndDate(xml);
				if (ts != 0 && ts < timestamp) {
					timestamp = ts;
				}
			}
			for (Iterator<?> iter = questionnaires.keys(); iter.hasNext(); ) {
				String title = (String) iter.next();
				String xml = questionnaires.optString(title);
				long ts = questionnaireEndDate(xml);
				if (ts != 0 && ts < timestamp) {
					timestamp = ts;
				}
			}
		}
		if (timestamp != Long.MAX_VALUE) {
			am.set(AlarmManager.RTC_WAKEUP, timestamp, piQM);
		}
	}

	// Sets ScheduledQuestionnaire alarm if repeating interval has passed
	private void setAlarmSQ() {
		AlarmManager am = (AlarmManager)getSystemService(Context.ALARM_SERVICE);

		long timestamp = Long.MAX_VALUE;
		String nextTitle = "";
		synchronized(questionnaires) {
			for (Iterator<?> iter = questionnaires.keys(); iter.hasNext(); ) {
				String title = (String) iter.next();
				String xml = questionnaires.optString(title);
				long ts = questionnaireStartDate(xml);
				long rt = questionnaireRepeatTime(xml);
				long now = System.currentTimeMillis();

				if (rt != 0){
					while (ts < now) {
						ts = ts + rt;
					}
					if (ts < timestamp) {
						timestamp = ts;
						nextTitle = title;
					}
				}
			}
		}

		if (timestamp != Long.MAX_VALUE) {
			Intent intentSQ = new Intent("com.questionnaire.ScheduledQuestionnaire");
			intentSQ.putExtra("title", nextTitle);
			PendingIntent piSQ = PendingIntent.getBroadcast(this, 0, intentSQ, PendingIntent.FLAG_CANCEL_CURRENT);
			am.set(AlarmManager.RTC_WAKEUP, timestamp, piSQ);
		}
	}




	@Override
	public void onCreate() {
		super.onCreate();
		Log.i("Service","onCreate");

		// Starts KeepAliveReceiver
		AlarmManager service = (AlarmManager) getSystemService(Context.ALARM_SERVICE);
		Intent i = new Intent(this, KeepAliveReceiver.class);
		PendingIntent pending = PendingIntent.getBroadcast(this, 0, i,
				PendingIntent.FLAG_CANCEL_CURRENT);

		service.setInexactRepeating(AlarmManager.RTC_WAKEUP,
				System.currentTimeMillis(), 60 * 1000, pending);


		// Creates user ID and stores it in shared preferences
		SharedPreferences prefs = getSharedPreferences("QuestionnairePreferences", MODE_PRIVATE);  
		if (prefs.contains("userId")) {
			userId = prefs.getString("userId", "");
		} else {
			final SecureRandom random = new SecureRandom();
			userId = new BigInteger(130, random).toString(32);

			SharedPreferences.Editor prefEditor = prefs.edit();  
			prefEditor.putString("userId", userId);  
			prefEditor.commit();  
		}

		// Loads questionnaires from assets or cache
		loadQuestionnaires();

		// Screen receiver setup
		IntentFilter filter = new IntentFilter();
		filter.addAction(Intent.ACTION_SCREEN_ON);
		filter.addAction(Intent.ACTION_SCREEN_OFF);
		registerReceiver(new ScreenReceiver(), filter);

		// Sets up all alarms
		IntentFilter ifilter = new IntentFilter();
		ifilter.addAction("com.questionnaire.ManualUploadReminder");
		ifilter.addAction("com.questionnaire.ScheduledQuestionnaire");
		ifilter.addAction("com.questionnaire.PeriodicDownloader");
		ifilter.addAction("com.questionnaire.QuestionnaireMover");
		ifilter.addAction("com.questionnaire.UploadManual");
		ifilter.addAction("com.questionnaire.SaveLog");
		registerReceiver(new AlarmReceiver(), ifilter);

		// Each of these four alarms is unique
		piMUR = PendingIntent.getBroadcast(this, 0, new Intent("com.questionnaire.ManualUploadReminder"), PendingIntent.FLAG_CANCEL_CURRENT);
		piPD = PendingIntent.getBroadcast(this, 0, new Intent("com.questionnaire.PeriodicDownloader"), PendingIntent.FLAG_CANCEL_CURRENT);
		piQM = PendingIntent.getBroadcast(this, 0, new Intent("com.questionnaire.QuestionnaireMover"), PendingIntent.FLAG_CANCEL_CURRENT);
		piSL = PendingIntent.getBroadcast(this, 0, new Intent("com.questionnaire.SaveLog"), PendingIntent.FLAG_CANCEL_CURRENT);

		AlarmManager am = (AlarmManager)getSystemService(Context.ALARM_SERVICE);
		am.setRepeating(AlarmManager.RTC_WAKEUP, System.currentTimeMillis(), pdTime*1000, piPD);
		am.setRepeating(AlarmManager.RTC_WAKEUP, System.currentTimeMillis(), slTime*1000, piSL);
		setAlarmMUR();
		setAlarmQM();
		setAlarmSQ();

		// Gets GPS location
		LocationManager locationManager = (LocationManager) this.getSystemService(Context.LOCATION_SERVICE);
		Criteria criteria = new Criteria();
		criteria.setAccuracy(Criteria.ACCURACY_COARSE);
		criteria.setAltitudeRequired(false);
		criteria.setBearingRequired(false);
		criteria.setPowerRequirement(Criteria.POWER_LOW);

		String provider = locationManager.getBestProvider(criteria, false);
		locationManager.requestLocationUpdates(provider, 60000, 10, locationListener);
		location = locationManager.getLastKnownLocation(provider);

		// Starts logger
		if (loggerThread == null) {
			loggerThread = new Thread(new LoggerRunnable());
			loggerThread.start();
		}
	}

	@Override
	public int onStartCommand(Intent intent, int flags, int startId) {
		Log.d("Service","onStartCommand");
		return Service.START_STICKY;
	}

	@Override
	public IBinder onBind(Intent intent) {
		Log.d("Service", "onBind");
		return inMessenger.getBinder();
	}

	@Override
	public void onRebind(Intent intent) {
		Log.d("Service", "onRebind");
	}

	@Override
	public boolean onUnbind(Intent intent) {
		Log.d("Service", "onUnbind");
		return true;
	}

	@Override	
	public void onDestroy() {
		Log.d("Service","onDestroy");
		saveLogObject();
		super.onDestroy();
	}

	// Handles incoming messages from QuestionnaireChooser and QuestionnaireActivity
	private class IncomingHandler extends Handler {
		@Override
		public void handleMessage(Message msg) {
			Bundle data = msg.getData();
			String action = data.getString("ACTION");

			Message backMsg = Message.obtain();
			backMsg.arg1 = Activity.RESULT_OK;

			// Starts function createChooserData after request from QuestionnaireChooser
			if (action.equals("getQuestionnaire")) {
				chooserMessenger = msg.replyTo;
				Bundle bundle = createChooserData();
				if (bundle == null) return;

				backMsg.setData(bundle);
				try {
					msg.replyTo.send(backMsg);
				} catch (android.os.RemoteException e1) {
					Log.w(getClass().getName(), "Exception sending message", e1);
				}
				
			// Triggers upload of answers after request by pushing upload answers button in options menu	
			} else if (action.equals("triggerUpload")) {
				new Thread(new Runnable() {
					@Override
					public void run() {
						uploadAnswers();
					}
				}).start();
				
			// Adds answered questionnaire together with user ID to JSON array answeredQuestionnaires,
			// increases answer count, updates variable last used and saves and uploads answers
			} else if (action.equals("newAnsweredQuestionnaire")) {
				SharedPreferences prefs = getSharedPreferences("QuestionnairePreferences", MODE_PRIVATE);  
				JSONObject newAnswerObject;
				try {
					newAnswerObject = new JSONObject(data.getString("newAnswerObject"));
					newAnswerObject.put("userId", userId);
				} catch (JSONException e) {
					newAnswerObject = new JSONObject();
				}

				answeredQuestionnaires.put(newAnswerObject);

				String title = newAnswerObject.optString("questionnaire");
				int count = answerCount.optInt(title);
				try {
					answerCount.put(newAnswerObject.optString("questionnaire"), count+1);
				} catch (JSONException e) {
					e.printStackTrace();
				}

				synchronized(logArray) {
					long timestamp = 0;
					try {
						lastUsed.put(questionnaireLogPackage(questionnaires.optString(title)), timestamp);
					} catch (JSONException e) {
						e.printStackTrace();
					}
				}

				SharedPreferences.Editor prefEditor = prefs.edit();  
				prefEditor.putString("answeredQuestionnaires", answeredQuestionnaires.toString());  
				prefEditor.putString("answerCount", answerCount.toString());  
				prefEditor.commit();  

				saveAnswerObjectToFile(newAnswerObject);

				SharedPreferences sharedPreferences = PreferenceManager.getDefaultSharedPreferences(QuestionnaireService.this);
				boolean autoUpload = sharedPreferences.getBoolean("pref_auto_upload_title", true);
				if (autoUpload == true) {
					new Thread(new Runnable() {
						@Override
						public void run() {
							uploadAnswers();
						}
					}).start();
				} else {
					setAlarmMUR();
				}

				moveQuestionnaires();
			} else {
				Log.i("Service", "Unknown message received!");
			}
		}
	}

	// Checks for running network connection dependent on user settings
	private boolean isNetworkAvailable() {
		ConnectivityManager cm = (ConnectivityManager) getSystemService(Context.CONNECTIVITY_SERVICE);
		NetworkInfo networkInfo = cm.getActiveNetworkInfo();
		if (networkInfo != null && networkInfo.isConnected()) {
			if (networkInfo.getType() == ConnectivityManager.TYPE_WIFI) {
				return true;
			} else  {
				SharedPreferences sharedPreferences = PreferenceManager.getDefaultSharedPreferences(this);
				boolean networkMode = sharedPreferences.getBoolean("pref_network_mode_title", true);
				return networkMode;
			}
		}
		return false;
	}

	// Loads available questionnaires
	private void loadQuestionnaires() {
		SharedPreferences prefs = getSharedPreferences("QuestionnairePreferences", MODE_PRIVATE);  
		// Loads available questionnaires from assets folder
		if (isReset || prefs.contains("questionnairesCache") == false) {	
			Log.i("Service", "Using Quests from assets");

			questionnaires = new JSONObject(); 
			updatedQuestionnaires = new JSONObject();
			answeredQuestionnaires = new JSONArray();
			answerCount = new JSONObject();
			lastUsed = new JSONObject();
			logArray = new JSONArray();

			// Saves the SysInfo questionnaire on first startup
			JSONObject sysInfo = new JSONObject();
			JSONObject newAnswerObject = new JSONObject();
			try {
				sysInfo.put("device", android.os.Build.DEVICE);
				sysInfo.put("version", android.os.Build.VERSION.RELEASE);
				sysInfo.put("apiLevel", android.os.Build.VERSION.SDK);
				sysInfo.put("model", android.os.Build.MODEL + " ("+ android.os.Build.PRODUCT + ")");
				WindowManager wm = (WindowManager) getSystemService(Context.WINDOW_SERVICE);
				Display display = wm.getDefaultDisplay();
				int w = display.getWidth();
				int h = display.getHeight(); 
				sysInfo.put("screen", Integer.toString(w) + "x" + Integer.toString(h));

				sysInfo.put("nfc", getPackageManager().hasSystemFeature(PackageManager.FEATURE_NFC));
				sysInfo.put("camera", getPackageManager().hasSystemFeature(PackageManager.FEATURE_CAMERA));
				sysInfo.put("gps", getPackageManager().hasSystemFeature(PackageManager.FEATURE_LOCATION_GPS));
				sysInfo.put("mode", mode.toString());

				newAnswerObject.put("timestamp", System.currentTimeMillis());
				newAnswerObject.put("questionnaire", "SystemInfo");
				newAnswerObject.put("answers", sysInfo);
				newAnswerObject.put("userId", userId);
			} catch (JSONException e) {
				e.printStackTrace();
			}
			answeredQuestionnaires.put(newAnswerObject);

			// Looks for available questionnaires
			Runnable runnable = new Runnable() {
				@Override
				public void run() {
					AssetManager mgr = getAssets();
					try {
						String list[] = mgr.list("questionnaires");

						for ( int i=0; i<list.length; i++) {
							InputStream in = mgr.open("questionnaires/" + list[i]);
							InputStreamReader is = new InputStreamReader(in);
							BufferedReader br = new BufferedReader(is);

							StringBuilder total = new StringBuilder();
							String line;
							while ((line = br.readLine()) != null) {
								total.append(line);
							}

							String title = questionnaireTitle(total.toString());
							synchronized(questionnaires) {
								updatedQuestionnaires.put(title, total.toString());
							}
						}
					} catch (Exception e) {
						e.printStackTrace();
					}
					
					// Updates shared preferences
					SharedPreferences prefs = getSharedPreferences("QuestionnairePreferences", MODE_PRIVATE);  
					SharedPreferences.Editor prefEditor = prefs.edit();
					prefEditor.putString("questionnairesCache", questionnaires.toString());
					prefEditor.putString("updatedQuestionnairesCache", updatedQuestionnaires.toString());
					prefEditor.putString("answeredQuestionnaires", answeredQuestionnaires.toString());
					prefEditor.putString("answerCount", answerCount.toString());
					prefEditor.putString("lastUsed", lastUsed.toString());
					prefEditor.putString("logArray", logArray.toString());
					prefEditor.putBoolean("welcomeScreenShown", false);
					prefEditor.putLong("lastUpdate", 0);
					prefEditor.commit();
				}
			};

			Thread t = new Thread(runnable);
			t.start();
			try {
				t.join();
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
		
		// Loads questionnaires and further variables from cache, 
		// starts updateMonitored to update monitored activities and package names
		} else {
			Log.i("Service", "Using Quests from cache");
			try {
				questionnaires = new JSONObject(prefs.getString("questionnairesCache", ""));
				updatedQuestionnaires = new JSONObject(prefs.getString("updatedQuestionnairesCache", ""));
				answeredQuestionnaires = new JSONArray(prefs.getString("answeredQuestionnaires", ""));
				answerCount = new JSONObject(prefs.getString("answerCount", ""));
				lastUsed = new JSONObject(prefs.getString("lastUsed", ""));
				logArray = new JSONArray(prefs.getString("logArray", ""));
			} catch (JSONException e) {
				e.printStackTrace();
			}
		}
		updateMonitored();
	}

	// Downloads new questionnaires from backend server, saves them into updatedQuestionnairesCache in shared preferences,
	// starts questionnaire mover and scheduled questionnaire alarm
	private void downloadQuestionnaires() {
		if (isNetworkAvailable() == false)
			return;

		try {
			SharedPreferences prefs = getSharedPreferences("QuestionnairePreferences", MODE_PRIVATE);  

			long timestamp = prefs.getLong("lastUpdate", 0);

			HttpParams httpParameters = new BasicHttpParams();

			HttpClient client = new DefaultHttpClient(httpParameters);
			HttpGet request = new HttpGet(serverIP + "/questionnaires/" + timestamp);

			HttpResponse response = client.execute(request);
			if (response.getStatusLine().getStatusCode() == 200) {
				BufferedReader rd = new BufferedReader(new InputStreamReader(response.getEntity().getContent()));

				String line = rd.readLine();
				JSONArray fromServer =  new JSONObject(line).optJSONArray("questionnaires");
				synchronized(questionnaires) {
					for (int i=0; i<fromServer.length(); i++) {
						String title = questionnaireTitle(fromServer.getString(i));
						updatedQuestionnaires.put(title, fromServer.getString(i));
					}

					SharedPreferences.Editor prefEditor = prefs.edit();  
					prefEditor.putString("updatedQuestionnairesCache", updatedQuestionnaires.toString());
					timestamp = new JSONObject(line).getLong("timestamp");
					prefEditor.putLong("lastUpdate", timestamp);
					prefEditor.commit();
				}

				setAlarmQM();
				setAlarmSQ();
			}
		} catch (Exception e) {
			e.printStackTrace();
		}		
	}

	// Uploads answers to backend server, which have not been uploaded yet, and triggers upload of logs
	// triggers manual upload reminder in case of upload failure
	private void uploadAnswers() {
		if (isNetworkAvailable() == false) {
			setAlarmMUR();
			return;
		}

		try {			
			HttpParams httpParameters = new BasicHttpParams();
			HttpClient client = new DefaultHttpClient(httpParameters);
			HttpPost request = new HttpPost(serverIP + "/answers");

			StringEntity se = new StringEntity(answeredQuestionnaires.toString(), HTTP.UTF_8);

			request.setEntity(se);
			request.setHeader("Accept", "application/json");
			request.setHeader("Content-Type", "application/json;charset=utf-8");

			HttpResponse response = client.execute(request);
			if (response.getStatusLine().getStatusCode() == 200) {
				Log.i("Service", "Answer upload completed successfully");
				SharedPreferences prefs = getSharedPreferences("QuestionnairePreferences", MODE_PRIVATE);
				SharedPreferences.Editor prefEditor = prefs.edit();
				answeredQuestionnaires = new JSONArray();
				prefEditor.putString("answeredQuestionnaires", answeredQuestionnaires.toString());
				prefEditor.commit();
				AlarmManager am = (AlarmManager)getSystemService(Context.ALARM_SERVICE);
				am.cancel(piMUR);
			} else {
				Log.i("Service", "Answer upload failed");
				setAlarmMUR();
				return;
			}
		} catch (Exception e) {
			Log.i("Service", "Answer upload timeout?");
			setAlarmMUR();
			return;
		}
		uploadLog();
	}

	// Adds questionnaires that reached their start date to list of available questionnaires, 
	// removes questionnaires that reached their end date from list of available questionnaires, 
	// removes questionnaires with a different group ID, removes questionnaires that reached their answer count
	private void moveQuestionnaires() {
		long currentTime = System.currentTimeMillis();
		synchronized(questionnaires) {
			for (Iterator<?> iter = updatedQuestionnaires.keys(); iter.hasNext(); ) {
				String title = (String) iter.next();
				String xml;
				try {
					xml = updatedQuestionnaires.getString(title);
					long startTime = questionnaireStartDate(xml);
					if (startTime<currentTime){
						Log.i("Service", "Updating/Activating " + title);
						questionnaires.put(title, xml);
						iter.remove();
					}
				} catch (JSONException e) {
					e.printStackTrace();
				}
			}
			for (Iterator<?> iter = questionnaires.keys(); iter.hasNext(); ) {
				String title = (String) iter.next();
				String xml;
				xml = questionnaires.optString(title);
				long enddate = questionnaireEndDate(xml);
				if (enddate > 0 && currentTime > enddate) {
					iter.remove();
					continue;
				}

				List<String> groupids = Arrays.asList(questionnaireGroupIDs(xml).replace(" ", "").split(","));
				if (groupids.contains(groupID) == false) {
					iter.remove();
					continue;
				}

				int maxCount = questionnaireRepeatNumber(xml);
				int currentCount = answerCount.optInt(title);
				if (maxCount > 0 && currentCount >= maxCount) {
					iter.remove();
					continue;
				}
			}

			// Saves updated questionnaire lists in shared preferences
			SharedPreferences prefs = getSharedPreferences("QuestionnairePreferences", MODE_PRIVATE);  
			SharedPreferences.Editor prefEditor = prefs.edit();
			prefEditor.putString("updatedQuestionnairesCache", new JSONObject().toString());
			prefEditor.putString("questionnairesCache", questionnaires.toString());
			prefEditor.commit();
		}
		updateMonitored();

		// Updates list of available questionnaires for QuestionnaireChooser
		if (chooserMessenger != null) {
			Message backMsg = Message.obtain();
			backMsg.arg1 = Activity.RESULT_OK;
			Bundle bundle = createChooserData();
			backMsg.setData(bundle);
			try {
				chooserMessenger.send(backMsg);
			} catch (android.os.RemoteException e1) {
				Log.w(getClass().getName(), "Exception sending message", e1);
			}
		}

		setAlarmQM();
		setAlarmSQ();
	}

	// Saves answers as txt file on SD card
	private void saveAnswerObjectToFile(JSONObject newAnswerObject) {
		final String answerString = newAnswerObject.toString();
		new Thread(new Runnable() {
			@Override
			public void run() {
				String state = Environment.getExternalStorageState();
				if (Environment.MEDIA_MOUNTED.equals(state)) {
					File dir = getExternalFilesDir(null);
					File answerFile = new File(dir, "AnsweredQuestionnaires.txt");
					try {
						FileWriter fw = new FileWriter(answerFile, true);
						fw.append(answerString + '\n');
						fw.flush();
					} catch (IOException e) {
						e.printStackTrace();
					}
				}						
			}
		}).start();
	}

	// Extracts questionnaire's title
	private String questionnaireTitle(String q) {
		Document doc = xmlDocumentFromString(q);
		return doc.getElementsByTagName("title").item(0).getTextContent().trim();
	}

	// Extracts questionnaire's start date
	private long questionnaireStartDate(String q) {
		Document doc = xmlDocumentFromString(q);
		long startdate = 0;
		try {
			SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd HH:mm");
			Date date = (Date)formatter.parse(doc.getElementsByTagName("startdate").item(0).getTextContent().trim());
			startdate = date.getTime();
		} catch (Exception e) {}
		return startdate;
	}

	// Extracts questionnaire's repeating interval
	private long questionnaireRepeatTime(String q) {
		if (mode != Mode.INTERVALTRIGGERED) return 0;

		Document doc = xmlDocumentFromString(q);
		long repeattime = 0;
		try {
			repeattime = Integer.parseInt(doc.getElementsByTagName("repeattime").item(0).getTextContent().trim());
		} catch (Exception e) {}
		return repeattime * 60 * 1000;
	}

	// Extracts questionnaire's repeating number
	private int questionnaireRepeatNumber(String q) {
		Document doc = xmlDocumentFromString(q);
		int repeatnumber = 0;
		try {
			repeatnumber = Integer.parseInt(doc.getElementsByTagName("repeatnumber").item(0).getTextContent().trim());
		} catch (Exception e) {}
		return repeatnumber;
	}

	// Extracts whether questionnaire's should autoopen or not
	private boolean questionnaireAutoOpen(String q) {
		if (mode != Mode.EVENTTRIGGERED) return false;

		Document doc = xmlDocumentFromString(q);
		boolean autoopen = false;
		try {
			autoopen = Integer.parseInt(doc.getElementsByTagName("autoopen").item(0).getTextContent().trim()) != 0;
		} catch (Exception e) {}
		return autoopen;
	}

	// Extracts questionnaire's group IDs
	private String questionnaireGroupIDs(String q) {
		Document doc = xmlDocumentFromString(q);
		String groupids = "";
		try {
			groupids = doc.getElementsByTagName("groupids").item(0).getTextContent().trim();
		} catch (Exception e) {}
		return groupids;
	}

	// Extracts questionnaire's end date
	private long questionnaireEndDate(String q) {
		Document doc = xmlDocumentFromString(q);
		long enddate = 0;
		try {
			SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd HH:mm"); 
			Date date = (Date)formatter.parse(doc.getElementsByTagName("enddate").item(0).getTextContent().trim());  
			enddate = date.getTime();
		} catch (Exception e) {}
		return enddate;
	}

	// Extracts questionnaire's logged activities
	private String questionnaireLogActivity(String q) {
		Document doc = xmlDocumentFromString(q);
		return doc.getElementsByTagName("logactivity").item(0).getTextContent().trim();
	}

	// Extracts questionnaire's logged package names
	private String questionnaireLogPackage(String q) {
		Document doc = xmlDocumentFromString(q);
		return doc.getElementsByTagName("logpackage").item(0).getTextContent().trim();
	}

	// Extracts information from xml 
	private Document xmlDocumentFromString(String s) {
		Document doc = null;
		try {
			DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
			DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
			InputSource is = new InputSource(new StringReader(s));
			doc = docBuilder.parse(is);
			doc.getDocumentElement().normalize();
		} catch (Exception e) {
			Log.i("Service", "XML parsing failed");
			e.printStackTrace();
		}
		return doc;
	}

	// Concatenates strings with no separator 
	private static String concatStringsWSep(List<String> strings, String separator) {
		StringBuilder sb = new StringBuilder();
		String sep = "";
		for(String s: strings) {
			sb.append(sep).append(s);
			sep = separator;
		}
		return sb.toString();                           
	}


}
